# _*_ coding: utf-8 _*_
# @Author : maoht
# @Date : 2024-07-22
# @Version :V0.01

import math
import numpy as np

PI = 3.1415926
degree2radian = PI / 180.0
# EARTH_RADIUS = 6371
EARTH_RADIUS = 6371137  # 地球平均半径
EARTH_RADIUS_LONG = 6378137.0  # 地球长半轴 单位：米
flattening = 1 / 298.257223563  # 扁率
EARTH_RADIUS_SHORT = EARTH_RADIUS_LONG - EARTH_RADIUS_LONG * flattening

def get_missile_loc(cur_loc, next_loc, speed, slope, intercept, durance):
    """
    根据导弹当前位置和飞行速度，方向，计算在一定时间后导弹的位置坐标
    :param cur_loc:当前坐标
    :param next_loc: 第一个拦截点坐标
    :param speed:飞行速度
    :param slope:直线斜率
    :param intercept: 直线截距
    :param durance:飞行时间
    :return:
    """
    dis = speed * durance
    # 两条直线的向量
    x = np.array([(next_loc[0] - cur_loc[0]), (next_loc[1] - cur_loc[1])])
    y = np.array([1, 0])
    # 模长
    length_x = math.sqrt(x.dot(x))
    length_y = math.sqrt(y.dot(y))
    # 根据向量之间求其夹角,求该直线方向与x轴的夹角
    theta = np.arccos(x.dot(y) / (float(length_x * length_y)))
    delta_x = dis * math.cos(theta)
    x2 = cur_loc[0] + delta_x
    y2 = slope * x2 + intercept
    return [x2, y2]

def get_horizontal_distance(geopoint1, geopoint2):
    """
    求两点的水平距离   Haversine公式
    :param geopoint1: tuple, (lat, lon), 例：(40.9, 140.0)
    :param geopoint2: tuple, (lat, lon), 例：(40.9, 142.0)
    :return: float, KM
    """
    lat1 = geopoint1[0] * degree2radian
    lon1 = geopoint1[1] * degree2radian
    lat2 = geopoint2[0] * degree2radian
    lon2 = geopoint2[1] * degree2radian

    difference = lat1 - lat2
    mdifference = lon1 - lon2
    distance = 2 * math.asin(math.sqrt(math.pow(math.sin(difference / 2), 2)
                                       + math.cos(lat1) * math.cos(lat2)
                                       * math.pow(math.sin(mdifference / 2), 2)))
    distance = distance * EARTH_RADIUS / 1000
    return distance

def get_geopoint_from_distance(geo_point, azimuth, distance_m):
    """
    从地球海拔水平上，选一角度出发一定距离后，获取新的点. 距离越远，精度越差
    :param geo_point: tuple, (float, float), (纬度, 经度)
    :param azimuth: float, 角度，0-360， 正北为0， 顺时针旋转360度
    :param distance_m: float, 距离，单位：m
    :return: tuple, (lat, lon)
    """
    lat = geo_point[0]
    lon = geo_point[1]
    a = EARTH_RADIUS_LONG
    b = EARTH_RADIUS_SHORT
    alpha1 = azimuth * PI / 180
    sinAlpha1 = math.sin(alpha1)
    cosAlpha1 = math.cos(alpha1)

    tanU1 = (1 - flattening) * math.tan(lat * PI / 180)
    cosU1 = 1 / math.sqrt((1 + tanU1 * tanU1))
    sinU1 = tanU1 * cosU1
    sigma1 = math.atan2(tanU1, cosAlpha1)
    sinAlpha = cosU1 * sinAlpha1
    cosSqAlpha = 1 - sinAlpha * sinAlpha
    uSq = cosSqAlpha * (a * a - b * b) / (b * b)
    A = 1 + uSq / 16384 * (4096 + uSq * (-768 + uSq * (320 - 175 * uSq)))
    B = uSq / 1024 * (256 + uSq * (-128 + uSq * (74 - 47 * uSq)))

    sigma = distance_m / (b * A)
    sigmaP = 2 * PI
    sinSigma = 0
    cosSigma = 0
    cos2SigmaM = 0
    for i in range(8):
        if math.fabs(sigma - sigmaP) < 1e-12:
            break
        cos2SigmaM = math.cos(2 * sigma1 + sigma)
        sinSigma = math.sin(sigma)
        cosSigma = math.cos(sigma)
        deltaSigma = B * sinSigma * (cos2SigmaM + B / 4 * (cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM)
                                                           - B / 6 * cos2SigmaM * (-3 + 4 * sinSigma * sinSigma) * (
                                                                   -3 + 4 * cos2SigmaM * cos2SigmaM)))
        sigmaP = sigma
        sigma = distance_m / (b * A) + deltaSigma

    tmp = sinU1 * sinSigma - cosU1 * cosSigma * cosAlpha1
    lat2 = math.atan2(sinU1 * cosSigma + cosU1 * sinSigma * cosAlpha1,
                      (1 - flattening) * math.sqrt(sinAlpha * sinAlpha + tmp * tmp))
    lon_span = math.atan2(sinSigma * sinAlpha1, cosU1 * cosSigma - sinU1 * sinSigma * cosAlpha1)
    C = flattening / 16 * cosSqAlpha * (4 + flattening * (4 - 3 * cosSqAlpha))
    lon_diff = lon_span - (1 - C) * flattening * sinAlpha * (
            sigma + C * sinSigma * (cos2SigmaM + C * cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM)))
    return lat2 * 180 / PI, lon + lon_diff * 180 / PI

def cal_azi(cur_loc, next_loc):
    """
    根据导弹当前位置和下一个目标位置，计算与正北方向的夹角,0-360
    :param cur_loc:
    :param next_loc:
    :return:
    """
    # 两条直线的向量
    x = np.array([(next_loc[1] - cur_loc[1]), (next_loc[0] - cur_loc[0])])
    y = np.array([0, 1])
    # 模长
    length_x = math.sqrt(x.dot(x))
    length_y = math.sqrt(y.dot(y))
    # 根据向量之间求其夹角,求该直线方向与正北的夹角
    azi = np.arccos(x.dot(y) / (float(length_x * length_y))) * 180/np.pi
    if next_loc[1] < cur_loc[1]:
        azi = 360 - azi
    return azi

def get_x_by_lon(base_lon, base_lat, lon):
    """
    :param base_lon: 初始参考经度110.595
    :param base_lat:21.036
    :param lon: 经度
    :return: 笛卡尔坐标系的x
    """
    # lon = round(lon, 5) - 0
    theta = lon - base_lon
    x = theta * math.pi * math.cos(base_lat * math.pi / 180) * EARTH_RADIUS / 180
    x = x * 1000
    return x

def get_y_by_lat(base_lat, lat):
    """
    :param base_lat: 初始参考维度
    :param lat: 维度
    :return: 笛卡尔坐标系的y
     """
    # lat = round(lat, 5) - 0
    theta = lat - base_lat
    y = theta * math.pi * EARTH_RADIUS / 180
    y = y * 1000
    return y

def get_usage_of_channel(allocation_result, ground_info_dic):
    """
    根据分配拦截结果，计算地导营通道的使用时间窗口
    :param allocation_result:武器分配结果
    :param ground_info_dic: 地导营信息
    :return:
    """
    usage_of_channel = dict()
    for key, value in allocation_result.items():
        usage_of_channel[key] = dict()
        for i in range(ground_info_dic[key]['fire_control_channel_num']):
            usage_of_channel[key][i] = list()
        sorted_missile_info = dict(sorted(value.items(), key=lambda x: x[1]['launch_time']))
        for id, info in sorted_missile_info.items():
            if info['trajectory_id'] not in usage_of_channel[key].keys():
                raise ValueError("请检查通道id是否错误！")
            usage_of_channel[key][info['trajectory_id']].append([info['launch_time'], info['explosion_time']])

    return usage_of_channel

def calculate_diagonal_length(side_length):
    """
    计算正方形的对角线长度，已知正方形的边长。
    :param side_length: 正方形的边长
    :return: 正方形的对角线长度
    """
    # 根据数学原理，对角线长度 = 边长 * sqrt(2)
    diagonal_length = side_length * math.sqrt(2)
    return diagonal_length

def calculate_endpoint(location, length, theta_degrees):
    """
    已知线段的起点、长度和与正北方向的夹角，计算线段终点的坐标。
    :param location:线段起点
    :param length:线段的长度
    :param theta_degrees:线段与正北方向的夹角
    :return:线段终点的坐标
    """
    # 将角度转换为与正东方向的夹角
    if theta_degrees <= 90:
        angle_with_east = 90 - theta_degrees
    angle_with_east = theta_degrees - 90
    # 将角度转换为弧度
    theta_radians = math.radians(angle_with_east)

    # 计算终点在 x 轴和 y 轴上的位移
    dx = length/111 * math.cos(theta_radians)
    dy = length/111 * math.sin(theta_radians)

    # 计算终点坐标
    end_x = location[0] + dx
    end_y = location[1] + dy

    return [end_x, end_y]

def divide_into_even_parts(total, parts):
    """
    数量分配函数，将一定数量分成近似均匀的几份
    :param total:待分配的数量
    :param parts:份数
    :return:分配结果，列表
    """
    base_part = total // parts
    remainder = total % parts
    parts_with_extra = [base_part + 1 for _ in range(remainder)]
    parts_without_extra = [base_part for _ in range(parts - remainder)]
    parts_with_extra.extend(parts_without_extra)
    return parts_with_extra

def circle_calculation(intercept_radius, missile_camp_loc):
    """
    已知圆心和半径，计算圆形区域
    :param intercept_radius:半径km
    :param missile_camp_loc:圆心位置坐标lat,lon
    :return:
    """
    # 生成角度值
    theta = np.linspace(0, 2 * np.pi, 100)

    # 计算圆上的x和y坐标
    x = intercept_radius/111 * np.cos(theta) + missile_camp_loc[1]
    y = intercept_radius/111 * np.sin(theta) + missile_camp_loc[0]

    return x, y
