# _*_ coding: utf-8 _*_
# @Author : MaoHT
# @Date : 2024-10-23
# @Version :V0.1
# description: 主函数入口
from discrete_simulation.time_window_calculation import *
from discrete_simulation.gro_missile_intercept_simulation import *
import simpy
from data.generate_data import *
from alg.greedy_allocation import *
from tools.common_functions import *
from situation_visualization import *

# 动态图表初始化
fig, (ax1, ax2) = plt.subplots(1, 2)


def update(frame):
    """
    更新函数
    :param frame: 帧数,即关键事件
    :return: 可视化结果
    """

    time = explosions[frame][1]  # 获取当前关键事件发生的时间
    ax1.cla()  # 子图清空
    ax2.cla()
    # 设置坐标轴范围和比例
    ax2.set_aspect('equal')
    ax2.set_title('Missile interception')  # 子图题目
    ax2.set_xlim(longitude_range[0] - 1, longitude_range[1] + 1)  # 设置膨胀范围，免得显示不全
    ax2.set_ylim(latitude_range[0] - 1, latitude_range[1] + 1)

    # 1、子图2地导营拦截目标情况可视化展示
    row_labels = []
    cell_text = []
    table_data = [['Unit', 'X_position', 'Y_position']]
    unit_info_new = copy.deepcopy(unit_info)
    threats = []
    current_positions = []
    # 遍历每个目标
    for target in explosions:
        # 1、更新表格中地导营的剩余弹药量
        if target[1] <= time:
            if target[9] is not None and target[10] is not None:
                unit_info_new[target[9]]['ammunition_amount'] -= target[10]
        # 2、计算每个关键事件发生时，其他目标所在的位置
        distance = time / 3600 * target[4] * 1000  # 距离单位为m，以导弹i的爆炸时间为一个节点，计算当前时间其他导弹的位置
        current_location = get_geopoint_from_distance(target[6], target[5], distance)
        # ax2.plot([target[6][1], target[7][1]], [target[6][0], target[7][0]], color='black', linestyle='--', lw='0.8')  # 预计进袭轨迹
        # 3、计算当前事件该目标是否到达爆炸点或者预定打击点，用不同颜色表示不同状态
        if target[9] is not None:
            if target[1] < time:  # 已经发生过的事件，跳过
                continue
            elif target[1] > time:  # 未发生的事件
                current_positions.append(current_location)
                threats.append(target[8])
                ax2.plot(target[7][1], target[7][0], color='blue', marker='<', markersize=2.5)  # 绘制打击目标点位置
            #     ax2.plot(current_location[1], current_location[0], color='b', marker='*')
            else:
                if target[3] == 'success':  # 地导营拦截目标成功
                    ax2.plot(target[2][1], target[2][0], color='g', marker='*', markersize=8)  # 绘制爆炸位置，绿色
                    ax2.plot([target[2][1], unit_info[target[9]]['loc'][1]],
                             [target[2][0], unit_info[target[9]]['loc'][0]],
                             '-r', lw='1')  # 地导营位置与爆炸位置连线
                elif target[3] == 'failed':  # 地导营拦截目标失败
                    ax2.plot(target[2][1], target[2][0], color='m', marker='*', markersize=8)  # 绘制爆炸位置，紫色
                    ax2.plot([target[2][1], unit_info[target[9]]['loc'][1]],
                             [target[2][0], unit_info[target[9]]['loc'][0]],
                             '-r', lw='1')  # 地导营位置与爆炸位置连线
                ax2.plot(target[7][1], target[7][0], color='blue', marker='<', markersize=2.5)  # 绘制打击目标位置
            x, y = curve_fitting(target[6], current_location, target[7])  # 曲线拟合
            ax2.plot(y, x, color='black', linestyle='--', linewidth='0.8')
        else:  # 地导营不能拦截目标
            if target[1] < time:  # 已经发生的事件，跳过
                continue
            elif target[1] > time:  # 未发生的事件
                current_positions.append(current_location)
                threats.append(target[8])
                ax2.plot(target[7][1], target[7][0], color='blue', marker='<', markersize=2.5)  # 绘制打击目标的位置
            else:  # 打击到指定位置的时刻
                ax2.plot(current_location[1], current_location[0], color='b', marker='*')  # 绘制当前位置
                circle = Circle((current_location[1], current_location[0]), 0.2, edgecolor='red', color='red',
                                linewidth=2)  # 绘制打击范围
                ax2.add_patch(circle)
                ax2.plot(target[7][1], target[7][0], color='blue', marker='<', markersize=2.5)  # 绘制打击目标的位置
            x, y = curve_fitting(target[6], current_location, target[7])  # 曲线拟合
            ax2.plot(y, x, color='black', linestyle='--', linewidth='0.8')

    # 将每个目标的威胁值用不同颜色表示
    positions = np.array(current_positions)
    # if positions.shape[0] > 0:
    a = positions[:, 1]
    b = positions[:, 0]
    scatter = ax2.scatter(positions[:, 1], positions[:, 0], c=threats, cmap='Blues')

    cbar = ColorbarBase(ax2.inset_axes([1.05, 0.1, 0.05, 0.8]), cmap='Blues', orientation='vertical',
                        ticks=[0, 50, 100])
    cbar.set_label('Threat Value')

    # 地导营位置和拦截范围绘制
    for unit_id, unit in unit_info_new.items():
        ax2.plot(unit["loc"][1], unit["loc"][0], color='r', marker='d')  # 地导营位置标记
        ax2.text(unit["loc"][1], unit["loc"][0], f'{unit_id}')  # 地导营名称信息标记
        cell_text.append([unit['loc'], unit["ammunition_amount"]])  # 地导营位置坐标、剩余弹药量显示
        row_labels.append(unit_id)
        x, y = circle_calculation(unit["intercept_radius"], unit["loc"])  # 地导营拦截范围展示
        ax2.plot(x, y, color='r')
        table_data += [[unit_id] + [unit["loc"][0], unit["loc"][1]]]

    # 2、子图1地导营数据信息实时更新展示
    ax1.axis('off')
    # 创建一个表格
    table = ax1.table(cellText=cell_text,
                      rowLabels=row_labels,
                      colLabels=['地导营位置', '弹药数量'],
                      loc='center',
                      cellLoc='center')
    # table.add_cell(0, 0, text="", loc='center')
    # # 调整表格大小以适应figure
    # ax1.set_title('地导营位置和弹药数量信息')
    table.auto_set_font_size(False)
    table.set_fontsize(10)
    table.scale(1, 1)


def input_and_process_data(situation_info, agent_info):
    """
    各函数调用
    :param situation_info: 态势数据
    :param agent_info: WTA分配决策数据
    :return:
    """
    cur_time = situation_info['cur_time']
    # 1、时间窗口计算
    # missile_info_dic = intercept_time_window_calculation(cur_time, agent_info, ground_info_dic)
    # print(missile_info_dic)

    # 2、地导营拦截导弹具体时间计算
    final_allocation_result = dict()
    for key, value in agent_info.items():
        unit = situation_info['unit_info'][key]
        trajectory_utility = list()
        for i in range(unit['fire_control_channel_num']):
            trajectory_utility.append({'cur_time': cur_time})
        allocation_result = greedy_allocation(copy.deepcopy(situation_info), value, trajectory_utility,
                                              copy.deepcopy(unit), cur_time)  # 在分配决策时不更新弹量等态势信息
        final_allocation_result[key] = allocation_result

    return final_allocation_result


def is_target_end(cur_time, situation_info, simulation_result):
    """
    判断目标是否全部消失或者全部被拦截
    :param cur_time:当前时刻
    :param situation_info:
    :param simulation_result:
    :return:
    """

    not_disappeared_target = []  # 筛选出当前阶段还未消失的目标
    for target_id, value in situation_info['target_info'].items():
        if value['disappear_time'] > cur_time:
            not_disappeared_target.append(target_id)
    if len(not_disappeared_target) > 0:
        for unit_id, target_info in simulation_result.items():
            if target_info == []:
                continue
            for result in target_info:
                target_id = result['target_id']
                if result is None:
                    continue
                elif result['state'] == 'success' and target_id in not_disappeared_target:
                    not_disappeared_target.remove(target_id)
    if len(not_disappeared_target) <= 0:
        return True
    else:
        return False


def get_current_simulation_result(time, simulation_result):
    """
    筛选当前阶段导弹发射时间早于当前时间的仿真事件结果
    :param time: 当前时间
    :param simulation_result: 整个阶段的仿真结果
    :return:
    """
    cur_simulation_result = dict()
    for unit_id, info in simulation_result.items():
        cur_simulation_result[unit_id] = []
        for result in info:
            if result['launch_time'] <= time:
                cur_simulation_result[unit_id].append(result)

    return cur_simulation_result


def record_cur_simulation_result(final_simulation_result, cur_simulation_result):
    """
    将当前阶段的仿真事件和仿真结果记录在最终仿真结果中
    :param final_simulation_result:
    :param cur_simulation_result:
    :return:
    """
    for unit_id, info in cur_simulation_result.items():
        if unit_id in final_simulation_result.keys():
            for result in info:
                final_simulation_result[unit_id].append(result)
        else:
            final_simulation_result[unit_id] = info

    return final_simulation_result


def update_situation(cur_time, initial_situation_info, situation_info, cur_simulation_result):
    """
    动态场景，更新下一阶段的态势数据，筛选剩余哪些目标，其余剩余弹药量等数据在仿真过程已经更新

    :param cur_time:下一个阶段的起始时间
    :param initial_situation_info:初始完整态势
    :param situation_info:更新前的态势
    :param cur_simulation_result:当前阶段的仿真结果
    :return:
    """
    situation_info['cur_time'] = cur_time
    # 记录当前时间已经被成功拦截或者消失的目标id，或者地导营已经发射拦截工作，打击动作尚未结束，以上三种情况的目标应当从态势中删除不再重新规划
    delete_target_id_lst = []
    for unit_id, target_info in cur_simulation_result.items():
        if target_info == []:
            continue
        for result in target_info:
            if situation_info['target_info'][result['target_id']][
                'disappear_time'] <= cur_time:  # 导弹目标在当前阶段消失，永久删除导弹目标信息
                delete_target_id_lst.append(result['target_id'])
            else:
                if result['state'] == 'success' and result['explosion_time'] <= cur_time:
                    delete_target_id_lst.append(result['target_id'])
                elif result['state'] == 'failed' and result['explosion_time'] <= cur_time:
                    if result['target_id'] not in situation_info['target_info'].keys():  # 当前阶段打击失败的目标，但当前态势中缺失,需要重新添加进去
                        target_info = initial_situation_info['target_info'][result['target_id']]
                        situation_info[result['target_id']] = target_info
                elif result['explosion_time'] > cur_time:
                    delete_target_id_lst.append(result['target_id'])  # 当前阶段目标导弹仍在正常飞行，但已分配打击任务，打击结果未知，暂时将该目标从态势中删除

    # 根据目标id进行删除
    for target_id in delete_target_id_lst:
        del situation_info['target_info'][target_id]
    # 更新每个目标的当前位置坐标
    for target_id, target_info in situation_info['target_info'].items():
        distance = situation_info['period_duration'] / 60 / 60 * target_info[
            'velocity'] * 1000  # 距离单位为m ,每个动态阶段更新一次目标位置
        cur_missile_loc = get_geopoint_from_distance(target_info['location'], target_info['theta_degrees'], distance)
        situation_info['target_info'][target_id]['location'] = cur_missile_loc

    # 更新地导营弹药量态势信息
    for unit_id, info in cur_simulation_result.items():
        if info == []:
            continue
        for result in info:
            situation_info['unit_info'][unit_id]['ammunition_amount'] -= result['used_ammunition_amount']

    return delete_target_id_lst


def simulation_run(situation_info, allocation_result):
    # 仿真运行
    env = simpy.Environment()
    simulation_result = launch_missiles(situation_info, allocation_result)
    env.process(simulation(situation_info, env, allocation_result))
    env.run()
    return simulation_result  # 返回仿真事件结果


def simulation_result_print(final_simulation_result):
    """
    地导拦截离散仿真结果打印输出
    :param final_simulation_result: 最终仿真结果
    :return:
    """
    intercepted_num = 0
    for unit_id, info in final_simulation_result.items():
        if info == []:
            continue
        for missile in info:
            i = unit_id
            j = missile['target_id']
            missile_intercept_location = missile['explosion_loc']
            explosion_time = missile['explosion_time']
            # 地导营拦截导弹的情况，分为成功、失败和无法拦截三种情况
            if missile['state'] == 'success':
                intercepted_num += 1
                print(f'地导营{i}成功拦截{j}的导弹，拦截地点为: {missile_intercept_location}，拦截时间： %.2f' % explosion_time)
            elif missile['state'] == 'failed':
                print(f'地导营{i}拦截{j}的导弹失败，拦截时间： %.2f' % explosion_time)
            else:
                print(f'{j}的导弹已经被击落，地导营{i}无需拦截')


situation_class = 0  # 场景类别参数，可配置参数，0为静态单阶段分配，1为动态多阶段分配

# agent_decision = [[1,6,0,6,3,0,0,3,30,20,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],
#         [0,0,0,0,0,0,0,0,0,0,5,0,0,0,25,3,0,0,32,0,0,0,0,0,11,24,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],
#         [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],
#         [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,7,0,0,0,0,0,0,0,12,8,9,23,4,0,0,12,25,0,0,0,0,0,0,0,0,0,0],
#         [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,35,0,0,0,0,0,0,6,13,0,0,15,21]
#          ]
agent_decision = [
    [1, 6, 75, 6, 3, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 1, 2, 48, 0, 1, 3, 3, 1, 24, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 2, 0, 0, 0, 1,
     0, 1, 2, 0, 0, 3, 51, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 3, 16, 0, 0, 0, 0, 0, 0, 0]
]

if __name__ == '__main__':
    if situation_class == 0:  # 静态单阶段武器目标分配
        situation_info = generate_situation_info()  # 态势数据生成函数返回态势
        agent_info = generate_agent_info_alg(situation_info)
        # agent_info = generate_agent_info(situation_info)
        # agent_info = agent_decision_proccess(agent_decision, situation_info)  #todo  agent_decision 需要输入
        final_allocation_result = input_and_process_data(situation_info, agent_info)  # 根据态势获取相应信息
        simulation_result = simulation_run(situation_info, final_allocation_result)  # 仿真运行，返回结果
        # 仿真结果打印输出

        simulation_result_print(simulation_result)
        # 可视化函数
        target_info = situation_info["target_info"]
        unit_info = situation_info["unit_info"]
        explosions = visualization(situation_info, simulation_result)
        ani = FuncAnimation(fig, update, frames=len(explosions) + 1, interval=1000)
        plt.show(block=True)
        # print(simulation_result)
    elif situation_class == 1:  # 动态多阶段武器目标分配，动态态势数据包括起始时间，阶段时间间隔
        initial_situation_info = generate_situation_info()  # 态势数据生成函数返回态势
        cur_time = initial_situation_info['cur_time']
        final_simulation_result = dict()  # 动态场景最终的模拟仿真结果
        situation_info = copy.deepcopy(initial_situation_info)
        count = 1
        delete_target = []
        while True:  # 结束条件包括目标全部拦截、武器剩余弹药量为0
            agent_info = generate_agent_info_alg_dynamic(situation_info, delete_target)
            # agent_info = agent_decision_proccess(agent_decision, situation_info)   # todo  agent_decision 需要输入，每个阶段决策WTA智能体均要重新输出决策结果
            final_allocation_result = input_and_process_data(situation_info, agent_info)
            cur_time += situation_info['period_duration']
            simulation_result = simulation_run(situation_info, final_allocation_result)  # 仿真运行，返回仿真结果
            # 截取当前阶段的截止时间的仿真结果
            cur_simulation_result = get_current_simulation_result(cur_time, simulation_result)
            # print("第"+ str(count) + "个阶段仿真结果")
            # simulation_result_print(cur_simulation_result)
            # print("---------------------")
            # 记录当前阶段的仿真事件以及仿真结果
            final_simulation_result = record_cur_simulation_result(final_simulation_result, cur_simulation_result)
            target_end = is_target_end(cur_time, situation_info, simulation_result)
            total_ammunition_amount = 0
            for key, value in situation_info['unit_info'].items():
                total_ammunition_amount += value['ammunition_amount']  # 计算所有武器单元剩余弹药数量
            if total_ammunition_amount <= 0 or target_end is True:
                break
            # 筛选出当前阶段的态势，更新态势，进入下一个阶段
            delete_target_id_lst = update_situation(cur_time, initial_situation_info, situation_info,
                                                    cur_simulation_result)
            if delete_target_id_lst:
                pausebutton = 1
            for id in delete_target_id_lst:
                if id not in delete_target:
                    delete_target.append(id)
            count += 1
        simulation_result_print(final_simulation_result)
        target_info = initial_situation_info["target_info"]
        unit_info = initial_situation_info["unit_info"]
        explosions = visualization(initial_situation_info, final_simulation_result)
        ani = FuncAnimation(fig, update, frames=len(explosions) + 1, interval=1000)
        plt.show(block=True)

    else:
        raise ValueError("请检查输入的场景类别参数是否正确！")
