import numpy as np
import matplotlib.pyplot as plt

plt.rcParams['font.sans-serif'] = ['SimHei']
plt.rcParams["axes.unicode_minus"] = False


class BinaryMOPSO:
    def __init__(self, p_matrix, V, c, swarm_size=50, max_iter=10, inertia=0.4):
        """
                :param p_matrix: m x n矩阵，拦截概率
                :param V: n维向量，目标价值
                :param c: m维向量，武器最大使用次数
                """
        self.p = p_matrix
        self.V = V
        self.c = c
        self.m, self.n = p_matrix.shape
        self.swarm_size = swarm_size
        self.max_iter = max_iter
        self.inertia = inertia
        self.v_max = 1
        self.topology_radius = 2  # 局部拓扑半径
        self.mutation_rate = 0.1

        # 初始化粒子群
        self.swarm = np.random.uniform(-1, 1, (swarm_size, self.m * self.n))
        self.velocity = np.zeros((swarm_size, self.m * self.n))
        self.p_best = self.swarm.copy()
        self.p_best_values = np.array([self.calculate_fitness(x) for x in self.swarm])

        # 外部存档
        self.repository = self.swarm[self.pareto(self.p_best_values)]
        self.archive_size = 50
        self.history = []  # 画图：用于存储帕累托前沿历史

    # 激活函数
    def sigmoid(self, x):
        return 1 / (1 + np.exp(-x))

    # 处理约束，每一个导弹营只能打击一个目标，每一个目标只能被一个导弹营打击
    def repair_solution(self, prob_matrix):
        """修复解以满足双重约束：
        1. 每个武器最多攻击c[i]个目标（武器容量约束）
        2. 每个目标只能被一个武器攻击（目标唯一性约束）"""

        # 初始化分配矩阵和标记数组
        prob_matrix = prob_matrix.reshape(self.m, self.n)
        X = np.zeros_like(prob_matrix, dtype=int)
        # X = X.reshape(self.m, self.n)
        target_assigned = np.zeros(self.n, dtype=bool)

        # 按武器效能排序（优化拦截价值的关键）
        weapon_efficiency = np.argsort(-np.max(self.p, axis=1))  # 按最大拦截概率排序武器

        for i in weapon_efficiency:  # 按优先级处理高价值武器
            if self.c[i] <= 0:
                continue

            # 获取当前武器对各目标的综合价值评分
            target_scores = prob_matrix[i] * self.V  # 概率×目标价值

            # 按综合价值降序选择目标
            for j in np.argsort(-target_scores):
                if not target_assigned[j] and (X[i].sum() < self.c[i]):
                    X[i, j] = 1
                    target_assigned[j] = True

                if X[i].sum() == self.c[i]:
                    break  # 达到武器容量上限

        return X

    # 多目标适应度函数
    def calculate_fitness(self, particle):
        X_bin = (self.sigmoid(particle) > 0.5).astype(int)
        X_bin = self.repair_solution(X_bin)

        # 目标1: 总拦截价值
        value = 0
        for j in range(self.n):
            survival = 1.0
            for i in range(self.m):
                if X_bin[i, j]:
                    survival *= (1 - self.p[i, j])
            value += self.V[j] * (1 - survival)

        # 目标2: 使用的武器总数
        used = np.sum(X_bin)

        return np.array([-value, used])  # 假设需要最大化价值和最小化使用量

    def pareto(self, fitness):
        # 执行帕累托筛选，找出帕累托最优解的索引
        pareto_index = np.ones(fitness.shape[0], dtype=bool)
        for i in range(len(fitness)):
            for j in range(len(fitness)):
                if i == j:
                    continue
                if np.all(fitness[j] <= fitness[i]) and np.any(fitness[j] < fitness[i]):
                    pareto_index[i] = False
                    break
        return pareto_index

    def non_dominated_sort(self, fitness):
        """快速非支配排序，返回前沿层级（0为最优前沿）"""
        n = len(fitness)
        fronts = [[]]
        domination_counts = np.zeros(n)
        dominated_solutions = [[] for _ in range(n)]

        for i in range(n):
            for j in range(i + 1, n):
                # 判断i和j的支配关系
                if np.all(fitness[i] <= fitness[j]) and np.any(fitness[i] < fitness[j]):
                    dominated_solutions[i].append(j)
                    domination_counts[j] += 1
                elif np.all(fitness[j] <= fitness[i]) and np.any(fitness[j] < fitness[i]):
                    dominated_solutions[j].append(i)
                    domination_counts[i] += 1

            if domination_counts[i] == 0:
                fronts[0].append(i)

        current_front = 0
        while fronts[current_front]:
            next_front = []
            for i in fronts[current_front]:
                for j in dominated_solutions[i]:
                    domination_counts[j] -= 1
                    if domination_counts[j] == 0:
                        next_front.append(j)
            current_front += 1
            fronts.append(next_front)

        return fronts[:-1]  # 最后一层为空

    def crowding_distance(self, fitness, front):
        """计算前沿层中解的拥挤距离"""
        n = len(front)
        distance = np.zeros(n)
        num_objectives = fitness.shape[1]

        for obj in range(num_objectives):
            sorted_indices = np.argsort(fitness[front, obj])
            distance[sorted_indices[0]] = distance[sorted_indices[-1]] = np.inf

            norm = fitness[sorted_indices[-1], obj] - fitness[sorted_indices[0], obj]
            if norm == 0:
                continue

            for i in range(1, n - 1):
                distance[sorted_indices[i]] += (
                                                       fitness[sorted_indices[i + 1], obj] - fitness[
                                                   sorted_indices[i - 1], obj]
                                               ) / norm

        return distance

    def update_repository(self):
        # 合并当前解
        all_solutions = np.vstack([self.repository, self.swarm])
        all_fitness = np.array([self.calculate_fitness(x) for x in all_solutions])

        # 非支配排序
        fronts = self.non_dominated_sort(all_fitness)

        # 逐步填充存档
        new_repository = []
        remaining = self.archive_size

        for front in fronts:
            if len(front) <= remaining:
                new_repository.extend(all_solutions[front])
                remaining -= len(front)
            else:
                # 当前前沿需要截断，按拥挤距离选择
                distances = self.crowding_distance(all_fitness, front)
                selected = np.argsort(-distances)[:remaining]  # 选拥挤距离大的
                new_repository.extend(all_solutions[front][selected])
                break

        self.repository = np.array(new_repository)

    def select_gbest(self):
        # 从帕累托前沿中选择拥挤距离最大的解（保持多样性）
        if len(self.repository) == 0:
            return self.swarm[np.random.randint(self.swarm_size)]

        # 计算帕累托前沿解的适应度
        fits = np.array([self.calculate_fitness(x) for x in self.repository])

        # 计算拥挤距离
        crowding = np.zeros(len(self.repository))
        for dim in range(fits.shape[1]):
            # 按当前维度排序
            order = np.argsort(fits[:, dim])
            # 边界解有无限距离
            crowding[order[0]] = np.inf
            crowding[order[-1]] = np.inf
            # 内部解计算距离
            for i in range(1, len(order) - 1):
                crowding[order[i]] += (fits[order[i + 1], dim] - fits[order[i - 1], dim])

        # 选择拥挤距离最大的解
        return self.repository[np.argmax(crowding)]

    def run(self):
        for epoch in range(self.max_iter):
            print(epoch)
            # 根据粒子当前位置的标准差调整
            self.v_max = np.std(self.swarm, axis=0) * 2
            # 非对称学习因子
            c1 = 2.5 - 1.0 * (epoch / self.max_iter)
            c2 = 0.5 + 2.0 * (epoch / self.max_iter)
            for i in range(self.swarm_size):
                r1, r2 = np.random.rand(2)
                # 选择全局最优
                all_fitness = np.array([self.calculate_fitness(x) for x in self.repository])
                g_best = self.select_gbest()
                # g_best = self.repository[self.pareto(all_fitness)]   # 选择适应度最高的点

                self.velocity[i] = self.inertia * self.velocity[i] \
                                   + c1 * r1 * (self.p_best[i] - self.swarm[i]) \
                                   + c2 * r2 * (g_best - self.swarm[i])
                self.velocity[i] = np.clip(self.velocity[i], -self.v_max, self.v_max)
                self.swarm[i] += self.velocity[i]

                # 变异操作
                if np.random.rand() < self.mutation_rate:
                    self.swarm[i] += np.random.normal(0, 0.5 * (1 - (-1)))

                self.swarm[i] = np.array(
                    [np.clip(x, -1, 1) for x in self.swarm[i]])

            self.update_repository()

            # 更新个体最优
            for i in range(self.swarm_size):
                current_fit = self.calculate_fitness(self.swarm[i])
                if np.all(current_fit <= self.p_best_values[i]):
                    self.p_best[i] = self.swarm[i]
                    self.p_best_values[i] = current_fit

            current_front = [self.calculate_fitness(x) for x in self.repository]
            self.history.append(current_front)  # 记录当前迭代的帕累托前沿

            # 显示进度
            if epoch % 10 == 0:
                best = np.min([self.calculate_fitness(x) for x in self.repository], axis=0)
                print(f"Iter {epoch}, Best Value: {best[0]:.2f}")

            # 返回最优解
        best_idx = np.argmin([self.calculate_fitness(x) for x in self.repository], axis=0)
        best_prob = self.repository[best_idx[0]]
        best_result = self.repair_solution(best_prob)
        best_obj = self.calculate_fitness(best_prob)
        return best_result, best_obj


def visualize_optimization(solver):
    # 创建画布和坐标轴
    fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(18, 6))
    plt.suptitle("多目标优化结果分析", fontsize=14)

    # 颜色映射（不同迭代的前沿用不同颜色）
    colors = plt.cm.viridis(np.linspace(0, 1, len(solver.history)))

    # --- 绘制所有帕累托前沿（ax1）---
    ax1.set_title('帕累托前沿演化', fontsize=12)
    ax1.set_xlabel('武器使用数量', fontsize=10)
    ax1.set_ylabel('总拦截价值', fontsize=10)
    ax1.grid(True, alpha=0.3)

    # 绘制历史前沿（透明度渐变）
    for i, front in enumerate(solver.history):
        f1 = [-x[0] for x in front]  # 拦截价值（取负）
        f2 = [x[1] for x in front]  # 武器使用数量
        ax1.scatter(f2, f1,
                    c=[colors[i]],
                    alpha=0.5,
                    label=f'迭代 {i + 1}' if i % 10 == 0 else "")

    # 标注最优前沿
    final_front = solver.history[-1]
    final_f1 = [-x[0] for x in final_front]
    final_f2 = [x[1] for x in final_front]
    ax1.scatter(final_f2, final_f1,
                c='red',
                s=50,
                marker='*',
                label='最终前沿')

    ax1.legend(loc='upper right', fontsize=8)

    # --- 绘制收敛曲线（ax2）---
    ax2.set_title('目标函数收敛趋势', fontsize=12)
    ax2.set_xlabel('迭代次数', fontsize=10)
    ax2.set_ylabel('目标函数值', fontsize=10)
    ax2.grid(True, alpha=0.3)

    # 计算统计量
    avg_f1 = [np.mean([-x[0] for x in front]) for front in solver.history]
    best_f1 = [np.max([-x[0] for x in front]) for front in solver.history]
    avg_f2 = [np.mean([x[1] for x in front]) for front in solver.history]
    best_f2 = [np.min([x[1] for x in front]) for front in solver.history]

    # 绘制曲线
    ax2.plot(avg_f1, 'b--', lw=1, label='平均拦截价值')
    ax2.plot(best_f1, 'b-', lw=2, label='最优拦截价值')
    ax2.plot(avg_f2, 'r--', lw=1, label='平均武器使用')
    ax2.plot(best_f2, 'r-', lw=2, label='最小武器使用')

    ax2.legend(fontsize=8)

    plt.tight_layout()
    plt.show()

# if __name__ == "__main__":
#     # np.random.seed(42)
#     m, n = 10, 5  # 10个武器，5个目标
#     p_matrix = np.random.rand(m, n) * 0.8 + 0.1  # 拦截概率在0.1-0.9之间
#     V = np.random.randint(1, 10, n)  # 目标价值1-10
#     c = np.ones(m, dtype=int)  # 每个武器最多使用1次
#
#     # 运行算法
#     solver = BinaryMOPSO(p_matrix, V, c, swarm_size=20, max_iter=20)
#     best_solution,best_obj = solver.run()
#
#     visualize_optimization(solver)
#
#     print("初始拦截概率：\n", p_matrix)
#     print("目标价值：\n", V)
#     print("最终分配方案：\n", best_solution)
#     print("最终目标函数值：\n", best_obj)
