import math

def haversine_distance(lat1, lon1, lat2, lon2):
    """
    使用Haversine公式计算两个经纬度坐标点之间的球面距离
        :param lat1: 起点纬度
        :param lon1: 起点经度
        :param lat2: 终点纬度
        :param lon2: 终点经度
        :return: 两点间距离（单位：公里）
    """
    # 地球半径，单位：公里
    R = 6371.0

    # 角度转弧度
    phi1 = math.radians(lat1)
    phi2 = math.radians(lat2)
    d_phi = math.radians(lat2 - lat1)
    d_lambda = math.radians(lon2 - lon1)

    # Haversine 公式
    a = math.sin(d_phi / 2) ** 2 + math.cos(phi1) * math.cos(phi2) * math.sin(d_lambda / 2) ** 2
    c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))

    return R * c


def compute_path_length_geo(path):
    """
    计算路径长度
        :param path: 由经纬度坐标点组成的列表 [(lat1,lon1), (lat2,lon2)...]
        :return: 路径总长度（单位：公里）
    """
    if len(path) < 2:
        return 0.0

    total_distance = 0.0
    for i in range(1, len(path)):
        lat1, lon1 = path[i - 1]
        lat2, lon2 = path[i]
        total_distance += haversine_distance(lat1, lon1, lat2, lon2)

    return total_distance


from math import atan2, cos, sin, radians


def compute_normal_offset_point(pt1, pt2, offset_km):
    """
    计算线段pt1-pt2的中垂线方向偏移点
        :param pt1: 起点坐标 (lat, lon)
        :param pt2: 终点坐标 (lat, lon)
        :param offset_km: 法线方向偏移距离（单位：公里）
        :return: 偏移点坐标 (new_lat, new_lon)
    """
    # 计算两点中心点
    mid_lat = (pt1[0] + pt2[0]) / 2
    mid_lon = (pt1[1] + pt2[1]) / 2

    # 计算方向角（正北为0，顺时针）
    angle = atan2(pt2[0] - pt1[0], pt2[1] - pt1[1])  # 纬度先、经度后， 纬度差作y，经度差作x

    # 垂直方向（法线方向）
    normal_angle = angle + radians(90)  # 垂直方向偏移

    # 每度纬度约等于 111km，经度需按纬度缩放
    dlat = offset_km / 111.0  # 1°纬度≈111km
    dlon = offset_km / (111.0 * cos(radians(mid_lat)))  # 考虑经度缩放

    # 法线方向偏移
    offset_lat = dlat * sin(normal_angle)  # 纬度变化量
    offset_lon = dlon * cos(normal_angle)  # 经度变化量

    return (mid_lat + offset_lat, mid_lon + offset_lon)


def extend_all_paths_with_curved_detour(path_dic, length_dic, max_diff=5, offset_km=10):
    """
    中点出插入”之“字型延长线段 保证路径长度相同  达到共同到达的目的
    通过插入中垂线偏移点延长路径，使所有路径长度趋于一致
        :param path_dic: 原始路径字典 {guid: path}
        :param length_dic: 原始路径长度字典 {guid: length}
        :param max_diff: 允许的最大长度差异（单位：公里）
        :param offset_km: 单次插入的绕行距离（单位：公里）
        :return: 延长后的路径字典 {guid: new_path}
    """

    # 获取所有路径中的最大长度作为基准
    max_length = max(length_dic.values())
    extended_dic = {}

    for guid, path in path_dic.items():
        current_length = length_dic[guid]
        # 长度已达标的路径直接保留
        if abs(current_length - max_length) <= max_diff:
            extended_dic[guid] = path
            continue

        # 需要延长的路径进行迭代处理
        new_path = path.copy()

        while True:
            current_length = compute_path_length_geo(new_path)
            delta = max_length - current_length
            # 长度满足要求时退出循环
            if delta <= max_diff:
                break

            # 选择当前路径的中间段进行操作（保证绕行点均匀分布）
            insert_index = len(new_path) // 2  # 当路径长度不足时，总是在当前路径的中间段行操作，保证绕行点均匀分布
            pt1 = new_path[insert_index - 1]
            pt2 = new_path[insert_index]

            # 生成法线方向偏移点（形成"之"字形绕行）
            detour_point = compute_normal_offset_point(pt1, pt2, offset_km)

            #  插入新点并更新路径
            new_path = new_path[:insert_index] + [detour_point] + new_path[insert_index:]

        extended_dic[guid] = new_path

    return extended_dic
